<?php 
include 'include/header.php';

// Verify session data is complete
if (!isset($_SESSION['user_id'])) {
    session_destroy();
    header("Location: login.php");
    exit();
}

// Database connection
require_once '../config/conn.php';

// Fetch student data
$student_id = $_SESSION['user_id'];
$stmt = $conn->prepare("SELECT * FROM students WHERE id = ?");
$stmt->bind_param("i", $student_id);
$stmt->execute();
$result = $stmt->get_result();
$student = $result->fetch_assoc();

if (!$student) {
    // Handle case where student doesn't exist
    $_SESSION['error'] = "Student record not found";
    header("Location: dashboard.php");
    exit();
}

// Helper functions
function format_date($date) {
    if (empty($date)) return 'Never';
    return date('M j, Y g:i A', strtotime($date));
}

function get_faculty_name($faculty_id) {
    global $conn;
    if (empty($faculty_id)) return 'Not specified';
    $stmt = $conn->prepare("SELECT name FROM faculties WHERE id = ?");
    $stmt->bind_param("i", $faculty_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $faculty = $result->fetch_assoc();
    return $faculty ? $faculty['name'] : 'Not specified';
}

function get_department_name($department_id) {
    global $conn;
    if (empty($department_id)) return 'Not specified';
    $stmt = $conn->prepare("SELECT name FROM departments WHERE id = ?");
    $stmt->bind_param("i", $department_id);
    $stmt->execute();
    $result = $stmt->get_result();
    $department = $result->fetch_assoc();
    return $department ? $department['name'] : 'Not specified';
}

// Get login history
$login_stmt = $conn->prepare("SELECT * FROM login_history WHERE user_id = ? ORDER BY login_time DESC LIMIT 1");
$login_stmt->bind_param("i", $student_id);
$login_stmt->execute();
$login_result = $login_stmt->get_result();
$last_login = $login_result->fetch_assoc();
?>

<!-- Content Area -->
<main class="main-content flex-1 p-6  dark:bg-gray-900 overflow-y-auto">
    <!-- Success/Error Messages -->
  <?php if (isset($_SESSION['success'])): ?>
    <div class="mb-4 bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded relative" role="alert">
      <span class="block sm:inline"><?= $_SESSION['success'] ?></span>
      <span class="absolute top-0 bottom-0 right-0 px-4 py-3" onclick="this.parentElement.style.display='none'">
        <svg class="fill-current h-6 w-6 text-green-500" role="button" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20"><title>Close</title><path d="M14.348 14.849a1.2 1.2 0 0 1-1.697 0L10 11.819l-2.651 3.029a1.2 1.2 0 1 1-1.697-1.697l2.758-3.15-2.759-3.152a1.2 1.2 0 1 1 1.697-1.697L10 8.183l2.651-3.031a1.2 1.2 0 1 1 1.697 1.697l-2.758 3.152 2.758 3.15a1.2 1.2 0 0 1 0 1.698z"/></svg>
      </span>
    </div>
    <?php unset($_SESSION['success']); ?>
  <?php endif; ?>
  
  <?php if (isset($_SESSION['error'])): ?>
    <div class="mb-4 bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative" role="alert">
      <span class="block sm:inline"><?= $_SESSION['error'] ?></span>
      <span class="absolute top-0 bottom-0 right-0 px-4 py-3" onclick="this.parentElement.style.display='none'">
        <svg class="fill-current h-6 w-6 text-red-500" role="button" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20"><title>Close</title><path d="M14.348 14.849a1.2 1.2 0 0 1-1.697 0L10 11.819l-2.651 3.029a1.2 1.2 0 1 1-1.697-1.697l2.758-3.15-2.759-3.152a1.2 1.2 0 1 1 1.697-1.697L10 8.183l2.651-3.031a1.2 1.2 0 1 1 1.697 1.697l-2.758 3.152 2.758 3.15a1.2 1.2 0 0 1 0 1.698z"/></svg>
      </span>
    </div>
    <?php unset($_SESSION['error']); ?>
  <?php endif; ?>
  <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
    <!-- Profile Card -->
    <div class=" dark:bg-gray-800 p-6 border border-gray-100 dark:border-gray-700 lg:col-span-1">
      <div class="flex flex-col items-center">
        <!-- Profile Picture -->
        <div class="relative mb-4">
          <img src="<?= !empty($student['profile_picture']) ? htmlspecialchars($student['profile_picture']) : 'assets/images/default-avatar.png' ?>" 
               alt="Profile Picture" 
               class="w-32 h-32 rounded-full object-cover border-4 border-primary-100 dark:border-primary-800">
          <button class="absolute bottom-0 right-0 bg-primary-500 text-white p-2 rounded-full hover:bg-primary-600 transition"> 
          </button>
          <a href="photo_edit.php" class="text-center bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-gray-300 dark:hover:bg-gray-600 transition">
            <i class="fas fa-camera mr-1"></i> Update Photo
          </a>
        </div>
        
        <!-- Student Info -->
        <h2 class="text-xl font-bold dark:text-white text-center"><?= htmlspecialchars($student['full_name']) ?></h2>
        <p class="text-gray-500 dark:text-gray-400 text-sm mb-4">
          <?= !empty($student['program']) ? htmlspecialchars($student['program']) : 'Not specified' ?> - 
          Year <?= !empty($student['year_of_study']) ? htmlspecialchars($student['year_of_study']) : 'Not specified' ?>
        </p>
        
        <!-- Student Details -->
        <div class="w-full space-y-3 mt-4">
          <div class="flex justify-between">
            <span class="text-gray-500 dark:text-gray-400">Student ID:</span>
            <span class="font-medium dark:text-white"><?= htmlspecialchars($student['student_id']) ?></span>
          </div>
          <div class="flex justify-between">
            <span class="text-gray-500 dark:text-gray-400">Registration No:</span>
            <span class="font-medium dark:text-white"><?= htmlspecialchars($student['reg_number']) ?></span>
          </div>
          <div class="flex justify-between">
            <span class="text-gray-500 dark:text-gray-400">Email:</span>
            <span class="font-medium dark:text-white"><?= htmlspecialchars($student['email']) ?></span>
          </div>
          <div class="flex justify-between">
            <span class="text-gray-500 dark:text-gray-400">Phone:</span>
            <span class="font-medium dark:text-white"><?= !empty($student['phone']) ? htmlspecialchars($student['phone']) : 'Not specified' ?></span>
          </div>
          <div class="flex justify-between">
            <span class="text-gray-500 dark:text-gray-400">Last Login:</span>
            <span class="font-medium dark:text-white"><?= format_date($student['last_login']) ?></span>
          </div>
          <?php if ($last_login): ?>
          <div class="flex justify-between">
            <span class="text-gray-500 dark:text-gray-400">Last Device:</span>
            <span class="font-medium dark:text-white"><?= htmlspecialchars($last_login['device_info']) ?></span>
          </div>
          <div class="flex justify-between">
            <span class="text-gray-500 dark:text-gray-400">Last IP:</span>
            <span class="font-medium dark:text-white"><?= htmlspecialchars($last_login['ip_address']) ?></span>
          </div>
          <?php endif; ?>
        </div>
        
        <!-- Edit Profile Button -->
        <a href="profile_edit.php" class="w-full mt-6 text-center bg-primary-500 dark:bg-primary-600 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-primary-600 transition">
          <i class="fas fa-user-edit mr-1"></i> Edit Profile
        </a>
      </div>
    </div>

    <!-- Main Content Area -->
    <div class=" dark:bg-gray-800  p-6 border border-gray-100 dark:border-gray-700 lg:col-span-2">
      <!-- Personal Information Section -->
      <div class="mb-8">
        <h3 class="text-lg font-semibold mb-4 dark:text-white border-b pb-2 border-gray-200 dark:border-gray-700">Personal Information</h3>
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Full Name</label>
            <p class="font-medium dark:text-white"><?= htmlspecialchars($student['full_name']) ?></p>
          </div>
          <div>
            <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Gender</label>
            <p class="font-medium dark:text-white"><?= !empty($student['gender']) ? ucfirst($student['gender']) : 'Not specified' ?></p>
          </div>
          <div>
            <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Date of Birth</label>
            <p class="font-medium dark:text-white"><?= !empty($student['dob']) ? format_date($student['dob']) : 'Not specified' ?></p>
          </div>
          <div>
            <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Nationality</label>
            <p class="font-medium dark:text-white"><?= !empty($student['nationality']) ? htmlspecialchars($student['nationality']) : 'Not specified' ?></p>
          </div>
        </div>
      </div>
      
      <!-- Academic Information Section -->
      <div class="mb-8">
        <h3 class="text-lg font-semibold mb-4 dark:text-white border-b pb-2 border-gray-200 dark:border-gray-700">Academic Information</h3>
        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
          <div>
            <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Faculty</label>
            <p class="font-medium dark:text-white"><?= get_faculty_name($student['faculty_id']) ?></p>
          </div>
          <div>
            <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Department</label>
            <p class="font-medium dark:text-white"><?= get_department_name($student['department_id']) ?></p>
          </div>
          <div>
            <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Program</label>
            <p class="font-medium dark:text-white"><?= !empty($student['program']) ? htmlspecialchars($student['program']) : 'Not specified' ?></p>
          </div>
          <div>
            <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Year of Study</label>
            <p class="font-medium dark:text-white">Year <?= !empty($student['year_of_study']) ? htmlspecialchars($student['year_of_study']) : 'Not specified' ?></p>
          </div>
          <div>
            <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Semester</label>
            <p class="font-medium dark:text-white">Semester <?= !empty($student['semester']) ? htmlspecialchars($student['semester']) : 'Not specified' ?></p>
          </div>
          <div>
            <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Status</label>
            <p class="font-medium <?= $student['is_active'] ? 'text-green-500' : 'text-red-500' ?>">
              <?= $student['is_active'] ? 'Active' : 'Inactive' ?>
            </p>
          </div>
        </div>
      </div>
    </div>
  </div>
</main>

<?php include 'include/footer.php'; ?>