<?php 
include 'include/header.php';

// Verify session data is complete
if (!isset($_SESSION['vendor_id'])) {
    session_destroy();
    header("Location: vendor-login.php");
    exit();
}

// Database connection
require_once '../config/conn.php';

// Fetch vendor data
$vendor_id = $_SESSION['vendor_id'];
$stmt = $conn->prepare("SELECT * FROM vendors WHERE id = ?");
$stmt->bind_param("i", $vendor_id);
$stmt->execute();
$result = $stmt->get_result();
$vendor = $result->fetch_assoc();

if (!$vendor) {
    // Handle case where vendor doesn't exist
    $_SESSION['error'] = "Vendor record not found";
    header("Location:index.php");
    exit();
}

// Helper functions
function format_date($date) {
    if (empty($date)) return 'Never';
    return date('M j, Y g:i A', strtotime($date));
}

// Get login history
$login_stmt = $conn->prepare("SELECT * FROM login_history WHERE user_id = ? ORDER BY login_time DESC LIMIT 1");
$login_stmt->bind_param("i", $vendor_id);
$login_stmt->execute();
$login_result = $login_stmt->get_result();
$last_login = $login_result->fetch_assoc();
?>

<!-- Content Area -->
<main class="main-content flex-1 p-6 dark:bg-gray-900 overflow-y-auto">
    <!-- Success/Error Messages -->
    <?php if (isset($_SESSION['success'])): ?>
    <div class="mb-4 bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded relative" role="alert">
        <span class="block sm:inline"><?= $_SESSION['success'] ?></span>
        <span class="absolute top-0 bottom-0 right-0 px-4 py-3" onclick="this.parentElement.style.display='none'">
        <svg class="fill-current h-6 w-6 text-green-500" role="button" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20"><title>Close</title><path d="M14.348 14.849a1.2 1.2 0 0 1-1.697 0L10 11.819l-2.651 3.029a1.2 1.2 0 1 1-1.697-1.697l2.758-3.15-2.759-3.152a1.2 1.2 0 1 1 1.697-1.697L10 8.183l2.651-3.031a1.2 1.2 0 1 1 1.697 1.697l-2.758 3.152 2.758 3.15a1.2 1.2 0 0 1 0 1.698z"/></svg>
        </span>
    </div>
    <?php unset($_SESSION['success']); ?>
    <?php endif; ?>
    
    <?php if (isset($_SESSION['error'])): ?>
    <div class="mb-4 bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded relative" role="alert">
        <span class="block sm:inline"><?= $_SESSION['error'] ?></span>
        <span class="absolute top-0 bottom-0 right-0 px-4 py-3" onclick="this.parentElement.style.display='none'">
        <svg class="fill-current h-6 w-6 text-red-500" role="button" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 20 20"><title>Close</title><path d="M14.348 14.849a1.2 1.2 0 0 1-1.697 0L10 11.819l-2.651 3.029a1.2 1.2 0 1 1-1.697-1.697l2.758-3.15-2.759-3.152a1.2 1.2 0 1 1 1.697-1.697L10 8.183l2.651-3.031a1.2 1.2 0 1 1 1.697 1.697l-2.758 3.152 2.758 3.15a1.2 1.2 0 0 1 0 1.698z"/></svg>
        </span>
    </div>
    <?php unset($_SESSION['error']); ?>
    <?php endif; ?>
    
    <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
        <!-- Profile Card -->
        <div class="dark:bg-gray-800 p-6 border border-gray-100 dark:border-gray-700 lg:col-span-1">
            <div class="flex flex-col items-center">
                <!-- Profile Picture -->
                <div class="relative mb-4">
                    <img src="../uploads/<?= !empty($vendor['profile_picture']) ? htmlspecialchars($vendor['profile_picture']) : 'assets/images/default-avatar.png' ?>" 
                        alt="Profile Picture" 
                        class="w-32 h-32 rounded-full object-cover border-4 border-primary-100 dark:border-primary-800">
                    <div class="mt-2">
                        <a href="photo_edit.php" class="text-center bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-gray-300 dark:hover:bg-gray-600 transition">
                            <i class="fas fa-camera mr-1"></i> Update Photo
                        </a>
                    </div>
                </div>
                
                <!-- Vendor Info -->
                <h2 class="text-xl font-bold dark:text-white text-center"><?= htmlspecialchars($vendor['business_name']) ?></h2>
                <p class="text-gray-500 dark:text-gray-400 text-sm mb-4">
                    <?= !empty($vendor['business_type']) ? htmlspecialchars($vendor['business_type']) : 'Not specified' ?>
                </p>
                
                <!-- Vendor Details -->
                <div class="w-full space-y-3 mt-4">
                    <div class="flex justify-between">
                        <span class="text-gray-500 dark:text-gray-400">Vendor ID:</span>
                        <span class="font-medium dark:text-white"><?= htmlspecialchars($vendor['vendor_id']) ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-500 dark:text-gray-400">Email:</span>
                        <span class="font-medium dark:text-white"><?= htmlspecialchars($vendor['email']) ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-500 dark:text-gray-400">Phone:</span>
                        <span class="font-medium dark:text-white"><?= !empty($vendor['phone']) ? htmlspecialchars($vendor['phone']) : 'Not specified' ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-500 dark:text-gray-400">Last Login:</span>
                        <span class="font-medium dark:text-white"><?= format_date($vendor['last_login']) ?></span>
                    </div>
                    <?php if ($last_login): ?>
                    <div class="flex justify-between">
                        <span class="text-gray-500 dark:text-gray-400">Last Device:</span>
                        <span class="font-medium dark:text-white"><?= htmlspecialchars($last_login['device_info']) ?></span>
                    </div>
                    <div class="flex justify-between">
                        <span class="text-gray-500 dark:text-gray-400">Last IP:</span>
                        <span class="font-medium dark:text-white"><?= htmlspecialchars($last_login['ip_address']) ?></span>
                    </div>
                    <?php endif; ?>
                </div>
                
                <!-- Edit Profile Button -->
                <a href="profile_edit.php" class="w-full mt-6 text-center bg-primary-500 dark:bg-primary-600 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-primary-600 transition">
                    <i class="fas fa-user-edit mr-1"></i> Edit Profile
                </a>
                
                <!-- Change Password Button -->
                <a href="password_change.php" class="w-full mt-4 text-center bg-blue-500 dark:bg-blue-600 text-white px-4 py-2 rounded-lg text-sm font-medium hover:bg-blue-600 transition">
                    <i class="fas fa-lock mr-1"></i> Change Password
                </a>
            </div>
        </div>

        <!-- Main Content Area -->
        <div class="dark:bg-gray-800 p-6 border border-gray-100 dark:border-gray-700 lg:col-span-2">
            <!-- Business Information Section -->
            <div class="mb-8">
                <h3 class="text-lg font-semibold mb-4 dark:text-white border-b pb-2 border-gray-200 dark:border-gray-700">Business Information</h3>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Business Name</label>
                        <p class="font-medium dark:text-white"><?= htmlspecialchars($vendor['business_name']) ?></p>
                    </div>
                    <div>
                        <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Business Type</label>
                        <p class="font-medium dark:text-white"><?= !empty($vendor['business_type']) ? htmlspecialchars($vendor['business_type']) : 'Not specified' ?></p>
                    </div>
                    <div>
                        <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Tax ID</label>
                        <p class="font-medium dark:text-white"><?= !empty($vendor['tax_id']) ? htmlspecialchars($vendor['tax_id']) : 'Not specified' ?></p>
                    </div>
                    <div>
                        <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Registration Date</label>
                        <p class="font-medium dark:text-white"><?= format_date($vendor['created_at']) ?></p>
                    </div>
                </div>
            </div>
            
            <!-- Contact Information Section -->
            <div class="mb-8">
                <h3 class="text-lg font-semibold mb-4 dark:text-white border-b pb-2 border-gray-200 dark:border-gray-700">Contact Information</h3>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Contact Person</label>
                        <p class="font-medium dark:text-white"><?= !empty($vendor['contact_person']) ? htmlspecialchars($vendor['contact_person']) : 'Not specified' ?></p>
                    </div>
                    <div>
                        <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Email</label>
                        <p class="font-medium dark:text-white"><?= htmlspecialchars($vendor['email']) ?></p>
                    </div>
                    <div>
                        <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Phone</label>
                        <p class="font-medium dark:text-white"><?= !empty($vendor['phone']) ? htmlspecialchars($vendor['phone']) : 'Not specified' ?></p>
                    </div>
                    <div>
                        <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Address</label>
                        <p class="font-medium dark:text-white"><?= !empty($vendor['address']) ? htmlspecialchars($vendor['address']) : 'Not specified' ?></p>
                    </div>
                </div>
            </div>
            
            <!-- Account Status Section -->
            <div class="mb-8">
                <h3 class="text-lg font-semibold mb-4 dark:text-white border-b pb-2 border-gray-200 dark:border-gray-700">Account Status</h3>
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <div>
                        <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Verification Status</label>
                        <p class="font-medium <?= $vendor['is_verified'] ? 'text-green-500' : 'text-yellow-500' ?>">
                            <?= $vendor['is_verified'] ? 'Verified' : 'Not Verified' ?>
                        </p>
                    </div>
                    <div>
                        <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Approval Status</label>
                        <p class="font-medium <?= $vendor['is_approved'] ? 'text-green-500' : 'text-yellow-500' ?>">
                            <?= $vendor['is_approved'] ? 'Approved' : 'Pending Approval' ?>
                        </p>
                    </div>
                    <?php if ($vendor['is_approved']): ?>
                    <div>
                        <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Approved By</label>
                        <p class="font-medium dark:text-white"><?= !empty($vendor['approved_by']) ? htmlspecialchars($vendor['approved_by']) : 'System' ?></p>
                    </div>
                    <div>
                        <label class="block text-sm text-gray-500 dark:text-gray-400 mb-1">Approval Date</label>
                        <p class="font-medium dark:text-white"><?= format_date($vendor['approval_date']) ?></p>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</main>

<?php include 'include/footer.php'; ?>